using System;
using System.Diagnostics;

namespace Implab {
    class PromiseActionReaction<T> : IResolvable<T> {
        readonly Deferred m_next;

        readonly IDispatcher m_dispatcher;

        readonly Action<T, Deferred> m_fulfilled;

        readonly Action<Exception, Deferred> m_rejected;

        public IPromise Promise {
            get { return m_next.Promise; }
        }

        public PromiseActionReaction(Action<T, Deferred> fulfilled, Action<Exception, Deferred> rejected, Deferred next, IDispatcher dispatcher) {
            m_next = next;
            m_fulfilled = fulfilled;
            m_rejected = rejected;
            m_dispatcher = dispatcher;
        }

        public void Resolve(T result) {
            if (m_fulfilled != null) {
                if (m_dispatcher != null)
                    m_dispatcher.Enqueue(ResolveImpl, result);
                else
                    ResolveImpl(result);
            } else {
                m_next.Resolve();
            }
        }

        void ResolveImpl (T result) {
            m_fulfilled(result, m_next);
        }

        public void Reject(Exception error) {
            if (m_fulfilled != null) {
                if (m_dispatcher != null)
                    m_dispatcher.Enqueue(RejectImpl, error);
                else
                    RejectImpl(error);
            } else {
                m_next.Reject(error);
            }
        }

        void RejectImpl(Exception error) {
            m_rejected(error, m_next);
        }

        public static PromiseActionReaction<T> Create(Action<T> fulfilled, Action<Exception> rejected, IDispatcher dispatcher) {
            return new PromiseActionReaction<T>(
                fulfilled != null ? PromiseHandler.Create(fulfilled) : null,
                rejected != null ? PromiseHandler.Create(rejected) : null,
                new Deferred(),
                dispatcher
            );
        }

        public static PromiseActionReaction<T> Create(Func<T,IPromise> fulfilled, Action<Exception> rejected, IDispatcher dispatcher) {
            return new PromiseActionReaction<T>(
                fulfilled != null ? PromiseHandler.Create(fulfilled) : null,
                rejected != null ? PromiseHandler.Create(rejected) : null,
                new Deferred(),
                dispatcher
            );
        }

        public static PromiseActionReaction<T> Create(Action<T> fulfilled, Func<Exception, IPromise> rejected, IDispatcher dispatcher) {
            return new PromiseActionReaction<T>(
                fulfilled != null ? PromiseHandler.Create(fulfilled) : null,
                rejected != null ? PromiseHandler.Create(rejected) : null,
                new Deferred(),
                dispatcher
            );
        }

        public static PromiseActionReaction<T> Create(Func<T,IPromise> fulfilled, Func<Exception, IPromise> rejected, IDispatcher dispatcher) {
            return new PromiseActionReaction<T>(
                fulfilled != null ? PromiseHandler.Create(fulfilled) : null,
                rejected != null ? PromiseHandler.Create(rejected) : null,
                new Deferred(),
                dispatcher
            );
        }
    }
}